#!/usr/bin/env python3
"""
Show live status of prestack_kids.py for lenses.csv and lenses_random.csv:
- PID, state, elapsed, RSS
- ΔCPU user/system over 5s
- context switch deltas
- FITS read % (if the FD is still tracked)

Usage:  python3 scripts/prestack_status.py
"""
import os, time, shutil

FITS = "data/KiDS_DR4.1_ugriZYJHKs_SOM_gold_WL_cat.fits"

def pids_for(target):
    cmd = f"pgrep -f 'prestack_kids.py .*{target}'"
    return [int(p) for p in os.popen(cmd).read().split() if p.strip()]

def proc_exists(pid): return os.path.exists(f"/proc/{pid}")

def read_cpu(pid):
    # utime, stime in clock ticks
    with open(f"/proc/{pid}/stat") as f:
        fields = f.read().split()
        ut, st = int(fields[13]), int(fields[14])
        state = fields[2]
        start_time_ticks = int(fields[21])
    # mem & ctx switches
    rss = vsz = v = nv = None
    with open(f"/proc/{pid}/status") as f:
        for line in f:
            if line.startswith("VmRSS:"): rss = line.split()[1]  # kB
            elif line.startswith("VmSize:"): vsz = line.split()[1]
            elif line.startswith("voluntary_ctxt_switches"): v = int(line.split()[-1])
            elif line.startswith("nonvoluntary_ctxt_switches"): nv = int(line.split()[-1])
    # elapsed
    clk = os.sysconf(os.sysconf_names['SC_CLK_TCK'])
    with open("/proc/uptime") as f: uptime = float(f.read().split()[0])
    # /proc/<pid>/stat start time is ticks since boot
    start_sec = start_time_ticks / clk
    etime = max(0.0, uptime - start_sec)
    return ut, st, state, v, nv, rss, vsz, etime, clk

def find_fits_fd(pid):
    base = os.path.basename(FITS)
    for ent in os.listdir(f"/proc/{pid}/fd"):
        try:
            link = os.readlink(f"/proc/{pid}/fd/{ent}")
            if base in link:
                return ent
        except OSError:
            continue
    return None

def fits_read_pct(pid, fd):
    try:
        size = os.stat(FITS).st_size
        with open(f"/proc/{pid}/fdinfo/{fd}") as f:
            for line in f:
                if line.startswith("pos:"):
                    pos = int(line.split()[1]); break
        return 100.0 * pos / size
    except Exception:
        return None

def report_for(label, target):
    pids = pids_for(target)
    if not pids:
        print(f"[{label}] no process found for {target}")
        return
    pid = pids[0]
    if not proc_exists(pid):
        print(f"[{label}] PID {pid} vanished")
        return
    ut0, st0, state0, v0, nv0, rss0, vsz0, et0, clk = read_cpu(pid)
    time.sleep(5)
    ut1, st1, state1, v1, nv1, rss1, vsz1, et1, _ = read_cpu(pid)
    dut, dst = (ut1 - ut0)/clk, (st1 - st0)/clk
    dv, dnv = (v1 - v0), (nv1 - nv0)
    fd = find_fits_fd(pid)
    pct = fits_read_pct(pid, fd) if fd else None
    rss_mb = (int(rss1) / 1024.0) if rss1 else None
    vsz_mb = (int(vsz1) / 1024.0) if vsz1 else None
    print(f"[{label}] PID {pid} state={state1} elapsed={et1:,.0f}s "
          f"RSS={rss_mb:.1f}MB VSZ={vsz_mb:.1f}MB "
          f"ΔCPU_user={dut:.2f}s ΔCPU_sys={dst:.2f}s Δctx(v/nv)={dv}/{dnv} "
          f"{'(FITS %.1f%%)' % pct if pct is not None else ''}")

def main():
    # quick file presence check
    for fp in ["data/prestacked_stacks_lens.csv","data/prestacked_meta_lens.csv",
               "data/prestacked_stacks_rand.csv","data/prestacked_meta_rand.csv"]:
        print(f"[file] {fp}: {'exists' if os.path.exists(fp) else '—'}")
    report_for("LENS   ", "lenses.csv")
    report_for("RANDOMS", "lenses_random.csv")

if __name__ == "__main__":
    main()
